#ifndef PEACH_H
#define PEACH_H

#define SDL_MAIN_HANDLED
#include <SDL2/SDL.h>
#include <stdbool.h>
#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

// define/enum --------------------------------------------------------------@/	
#define PEACH_ERROR(msg) _peach_error(__LINE__,__FILE__,__func__,msg)
#define PEACH_ASSERT(cond,msg) { if(!(cond)) { PEACH_ERROR(msg); } }
#define PEACH_GRAPH_MAX (512)
#define PEACH_DRAWSTATE_MAX (64)

#define PEACH_DRAWFLIP_NONE (0)
#define PEACH_DRAWFLIP_H (0b01)
#define PEACH_DRAWFLIP_V (0b10)
#define PEACH_DRAWFLIP_HV (PEACH_DRAWFLIP_H | PEACH_DRAWFLIP_V)

enum PEACH_DRAWBLEND {
	PEACH_DRAWBLEND_NONE,
	PEACH_DRAWBLEND_ALPHA,
	PEACH_DRAWBLEND_ADD
};

// types --------------------------------------------------------------------@/
typedef struct CPeachResource {
	uint16_t alive;
	uint16_t id;
} CPeachResource;

typedef struct CPeachGraph {
	CPeachResource basersrc;
	int16_t size_x, size_y;
	SDL_Texture* texture_hnd;
} CPeachGraph;

typedef struct CPeachDrawstate {
	int trans_x, trans_y;
	int blendmode;
} CPeachDrawstate;

typedef struct CPeachPad {
	int16_t joy_x, joy_y;
} CPeachPad;

typedef struct CPeachColor {
	uint8_t r;
	uint8_t g;
	uint8_t b;
	uint8_t a;
} CPeachColor;

typedef struct CPeachWork {
	int window_sizeX,window_sizeY;
	SDL_Window* window_hnd;
	SDL_Renderer* renderer_hnd;
	int renderer_vsync;

	CPeachGraph table_graph[PEACH_GRAPH_MAX];
	CPeachPad table_pad[2];
	
	CPeachDrawstate drawstate;
	CPeachDrawstate drawstate_stack[PEACH_DRAWSTATE_MAX];
	int drawstate_idx;

	uint32_t time_lastFrame;
} CPeachWork;

// variable -----------------------------------------------------------------@/
extern CPeachWork* gPeachWork;

// function -----------------------------------------------------------------@/
static inline CPeachWork* peach_workGet() {
	return gPeachWork;
}
static inline CPeachDrawstate* peach_drawstateGet() {
	return &peach_workGet()->drawstate;
}
static inline CPeachColor peach_colorGetRGBA(int r, int g, int b, int a) {
	return (CPeachColor){
		.r = r,
		.g = g,
		.b = b,
		.a = a
	};
}
static inline CPeachColor peach_colorGetRGB(int r, int g, int b) {
	return peach_colorGetRGBA(r,g,b,255);
}

// sys.c ------------------------------------------------@/
void peach_init(int win_width, int win_height);
void peach_close();
bool peach_processMessage();
void _peach_error(int line, const char* filename, const char* funcname, const char* message);

// graph.c ----------------------------------------------@/
CPeachGraph* peach_graph_loadFile(const char* filename);
void peach_graph_close(CPeachGraph* gr);
void peach_graph_drawPart(CPeachGraph* gr, int sx, int sy, int sw, int sh, int dx, int dy, int flip);
void peach_graph_draw(CPeachGraph* gr, int x, int y, int flip);

// render.c ---------------------------------------------@/
CPeachDrawstate* peach_drawstateGet();
void peach_drawstate_pop();
void peach_drawstate_push();
void peach_drawstate_reset();
void peach_drawstate_translate(float x, float y);
void peach_drawstate_blendSet(int mode);
void peach_drawstate_blendReset();

void peach_draw_rect(CPeachColor color, int x, int y, int w, int h);
void peach_draw_clear(CPeachColor color);
void peach_draw_sync();

// pad.c ------------------------------------------------@/
void peach_padGet(CPeachPad* pad, size_t idx);
void peach_pad_updateAll();

#ifdef __cplusplus
}
#endif

#endif

