#include <peach.h>

void peach_drawstate_pop() {
	auto work = peach_workGet();
	PEACH_ASSERT(work->drawstate_idx >= 0,
		"drawstate stack underflow!"
	);
	
	work->drawstate = work->drawstate_stack[--work->drawstate_idx];

	// we call SDL_SetRenderDrawBlendMode again to reset
	// the blendmode to the previous one
	auto state = peach_drawstateGet();
	peach_drawstate_blendSet(state->blendmode);
}
void peach_drawstate_push() {
	auto work = peach_workGet();
	PEACH_ASSERT(work->drawstate_idx <= PEACH_DRAWSTATE_MAX,
		"drawstate stack overflow!"
	);

	auto state = peach_drawstateGet();
	work->drawstate_stack[work->drawstate_idx++] = *state;
}
void peach_drawstate_reset() {
	auto state = peach_drawstateGet();
	state->trans_x = 0;
	state->trans_y = 0;
	peach_drawstate_blendReset();
}
void peach_drawstate_translate(float x, float y) {
	auto state = peach_drawstateGet();
	state->trans_x += x;
	state->trans_y += y;
}
void peach_drawstate_blendSet(int mode) {
	auto work = peach_workGet();
	auto state = peach_drawstateGet();
	int real_blendmode = 0;

	// get SDL blend mode enum --------------------------@/
	switch(mode) {
		case PEACH_DRAWBLEND_ALPHA: real_blendmode = SDL_BLENDMODE_BLEND; break;
		case PEACH_DRAWBLEND_ADD: real_blendmode = SDL_BLENDMODE_ADD; break;
		case PEACH_DRAWBLEND_NONE: real_blendmode = SDL_BLENDMODE_NONE; break;
		default: {
			PEACH_ERROR("invalid blend mode!");
			break;
		}
	}

	state->blendmode = mode;
	SDL_SetRenderDrawBlendMode(work->renderer_hnd,real_blendmode);
}
void peach_drawstate_blendReset() {
	peach_drawstate_blendSet(PEACH_DRAWBLEND_ALPHA);
}

void peach_draw_rect(CPeachColor color, int x, int y, int w, int h) {
	auto work = peach_workGet();
	
	// set color ----------------------------------------@/
	SDL_SetRenderDrawColor(work->renderer_hnd,
		color.r,color.g,color.b,color.a
	);

	// draw rect ----------------------------------------@/
	auto state = peach_drawstateGet();
	const SDL_Rect draw_rect = {
		.x = x + state->trans_x,
	   	.y = y + state->trans_y,
		.w = w, .h = h
	};

	SDL_RenderFillRect(work->renderer_hnd,&draw_rect);
}
void peach_draw_clear(CPeachColor color) {
	auto work = peach_workGet();
	SDL_SetRenderDrawColor(work->renderer_hnd,
		color.r,color.g,color.b,color.a
	);
	SDL_RenderClear(work->renderer_hnd);
}
void peach_draw_sync() {
	auto work = peach_workGet();

	if(work->renderer_vsync) {
		SDL_RenderPresent(work->renderer_hnd);
	} else {
		SDL_RenderPresent(work->renderer_hnd);
		// busy-wait until next frame -------------------@/
		// wait for a portion of a frame, then busy-wait out the end, just
		// to be super precise.
		int next_frameendPeriod1 = work->time_lastFrame + 14;
		int next_frameendPeriod2 = work->time_lastFrame + 16;
		uint64_t busywait_amt = 0;
		while(SDL_GetTicks() < next_frameendPeriod1) {
			SDL_Delay(1);
		}
		while(SDL_GetTicks() < next_frameendPeriod2) {
			busywait_amt++;
		}
		work->time_lastFrame = SDL_GetTicks();
		printf("busy wait times: %llu\n",busywait_amt);
	}
}

