#include <pgame.h>

// variable -----------------------------------------------------------------@/
PGame* gPGame = NULL;

// function -----------------------------------------------------------------@/
void pgame_setup() {
	// initialize peach ---------------------------------@/
	peach_init(SCREEN_X,SCREEN_Y);

	// initailize game ----------------------------------@/
	PGame* work = calloc(sizeof(PGame),1);
	gPGame = work;

	pgame_player_setup();
	pgame_ball_reset();

	work->stateID = GAMESTATE_INVALID;
	work->stateIDNew = GAMESTATE_BALLWAIT;

	work->time_frame = 0;
	work->time_seconds = 0;
	work->time_mins = 0;
	work->timer_active = false;

	// initialize resources -----------------------------@/
	work->gr_fontPuchi = peach_graph_loadFile("data/img/puchifnt.png");
	work->gr_fontScore = peach_graph_loadFile("data/img/scorefnt.png");
	work->gr_fieldbg = peach_graph_loadFile("data/img/fieldbg.png");
}
void pgame_close() {
	// close assets -------------------------------------@/
	auto work = pgame_workGet();
	peach_graph_close(work->gr_fontPuchi);
	peach_graph_close(work->gr_fontScore);
	peach_graph_close(work->gr_fieldbg);

	// free game ----------------------------------------@/
	free(gPGame);
	gPGame = NULL;

	// close peach --------------------------------------@/
	peach_close();
}
void pgame_start() {
	pgame_setup();

	while(peach_processMessage()) {
		pgame_update();
		pgame_draw();
	}

	pgame_close();
}

void pgame_update() {
	peach_pad_updateAll();

	// check for state changes --------------------------@/
	auto work = pgame_workGet();
	if(work->stateIDNew != GAMESTATE_INVALID) {
		int newid = work->stateIDNew;
		work->stateID = newid;
		work->stateIDNew = GAMESTATE_INVALID;
		if(newid == GAMESTATE_BALLWAIT) work->wait_timer = WAITTIMER_BALLWAIT;
		if(newid == GAMESTATE_WIN) work->wait_timer = WAITTIMER_WIN;
	}

	// update current state -----------------------------@/
	switch(work->stateID) {
		case GAMESTATE_BALLWAIT: {
			static const int plr_xpos = ((FIELD_X/2) - (PADDLE_SIZEX/2)) << 8;
			work->ball_canDraw = true;
			work->ball_canMove = false;
			work->player_canMove = false;
			work->timer_active = false;
			pgame_playerGet(0)->pos.x = plr_xpos;
			pgame_playerGet(1)->pos.x = plr_xpos;
			pgame_ball_reset();

			// decrement timer
			work->wait_timer--;
			if(work->wait_timer == 0) {
				work->stateIDNew = GAMESTATE_PLAYING;
				work->time_frame = 0;
				work->time_seconds = 0;
				work->time_mins = 0;

				// ball should move upwards if owner is P1,
				// or downwards if P2
				work->ball.mov.x = 0;
				if(work->ball.owner == 0) {
					work->ball.mov.y = -1 * BALL_SPEED;
				} else {
					work->ball.mov.y = BALL_SPEED;
				}
			}
			break;
		}
		case GAMESTATE_PLAYING: {
			work->ball_canDraw = true;
			work->ball_canMove = true;
			work->player_canMove = true;
			work->timer_active = true;
			break;
		}
		case GAMESTATE_WIN: {
			work->ball_canDraw = false;
			work->ball_canMove = false;
			work->player_canMove = false;
			work->timer_active = false;
			work->score_wintimer++;
			work->flash_timer++;
			pgame_ball_reset();

			// decrement timer
			work->wait_timer--;
			if(work->wait_timer == 0) {
				work->stateIDNew = GAMESTATE_BALLWAIT;
			}
			break;
		}
	}

	pgame_player_updateAll();
	pgame_ball_update();

	// check round timer --------------------------------@/
	if(work->timer_active) {
		work->time_frame++;
		if(work->time_frame >= 60) {
			work->time_frame = 0;
			work->time_seconds += 1;
		}
		if(work->time_seconds >= 60) {
			work->time_seconds = 0;
			work->time_mins += 1;
		}
		if(work->time_mins >= 99) work->time_mins = 99;
	}
	work->time_frameAbsolute++;
}
void pgame_draw() {
	peach_draw_clear(peach_colorGetRGB(240,240,240));
	
	// draw field background, then player & ball
	peach_graph_draw(pgame_workGet()->gr_fieldbg,0,0,PEACH_DRAWFLIP_NONE);
	pgame_ball_draw();
	pgame_player_drawAll();
	pgame_drawUI();

	peach_draw_sync();
}

void pgame_ball_reset() {
	auto work = pgame_workGet();
	auto ball = &work->ball;

	ball->pos = (vec2i) { (FIELD_X/2) << 8, (FIELD_Y/2) << 8 };
	ball->mov = (vec2i) { 0,0 };
	ball->did_finish = false;
}
void pgame_ball_update() {
	auto work = pgame_workGet();
	auto ball = &work->ball;

	if(!work->ball_canMove) return;

	int mx = ball->mov.x;
	int my = ball->mov.y;
	int px = ball->pos.x;
	int py = ball->pos.y;

	px += mx;
	py += my;

	// check for field collision ------------------------@/
	static const int field_x = FIELD_X << 8;
	static const int field_y = FIELD_Y << 8;

	if(px >= field_x) {
		px = field_x;
		mx = -mx;
	}
	if(px < 0) {
		px = 0;
		mx = -mx;
	}

	// check for player collision -----------------------@/
	// but, only do it if ball didn't already hit.
	if(!ball->did_finish) {
		for(int i=0; i<2; i++) {
			if(ball->owner == i) continue;
			const auto player = pgame_playerGet(i);
			const vec2i paddlesize = { PADDLE_SIZEX<<8, PADDLE_SIZEY<<8 };
			const vec2i* plr_pos = &player->pos;
			const int plr_xCenter = plr_pos->x + (paddlesize.x/2);
			
			// if ball's in the range of the two paddles' vertical axes...
			if(py < paddlesize.y || py >= field_y-paddlesize.y) {
				// if ball's in range of one of the paddles' width...
				if(px >= plr_pos->x && px < plr_pos->x + paddlesize.x) {
					int dist_center = (px - plr_xCenter);
					mx = dist_center / 8;
					// Y velocity should be set to (-y) * 1.02
					// however, multiplication by two fixed-point numbers
					// is (a * b) >> shift.
					my = ((-my) * (int)(256 * 1.02)) >> 8;
					ball->owner = i;
					if(i == 0) py = (field_y-paddlesize.y)-1;
					if(i == 1) py = paddlesize.y;
					player->shake_timer = PLAYER_SHAKEDURATION;
				}
			}
		}
	}

	// P2 win case
	if(py >= field_y) {
		ball->did_finish = true;
		ball->owner = 1;
	}

	// P1 win case
	if(py < 0) {
		// p1 win
		ball->owner = 0;
		ball->did_finish = true;
	}



	ball->pos = (vec2i) { px,py };
	ball->mov = (vec2i) { mx,my };

	// check if ball finished ---------------------------@/
	if(ball->did_finish) {
		pgame_playerGet(ball->owner)->score += 1;
		ball->did_finish = false;
		work->stateIDNew = GAMESTATE_WIN;
		work->score_wintimer = 0;
		work->flash_timer = 0;
	}
}
void pgame_ball_draw() {
	auto work = pgame_workGet();
	if(!work->ball_canDraw) return;
	const auto ball = &work->ball;

	int ball_posX = (ball->pos.x >> 8);
	int ball_posY = (ball->pos.y >> 8);
	peach_draw_rect(peach_colorGetRGB(16,16,16),
		ball_posX-4,ball_posY-4,
		8,8
	);
}

void pgame_drawUI() {
	const auto work = pgame_workGet();
	char textbufP1[64];
	char textbufP2[64];
	char textbufRoundtime[64];

	sprintf(textbufP1,"%3d", pgame_playerGet(0)->score);
	sprintf(textbufP2,"%3d", pgame_playerGet(1)->score);

	peach_drawstate_push();
	peach_drawstate_translate(SUBFIELD_XPOS,SUBFIELD_YPOS);
	
	// draw score ---------------------------------------@/
	if(work->stateID == GAMESTATE_WIN) {
		// do jolting animation with score --------------@/
		// each entry in wintimer_lut acts as a keyframe,
		// storing the Y offset to use on every frame.
		static const int wintimer_lut[] = {
			-6,-3,2,2,
			-1,0,1,0
		};
		static const size_t wintimer_len = sizeof(wintimer_lut) / sizeof(int);
		int wintimer_idx = work->score_wintimer/2;
		if(wintimer_idx >= wintimer_len) wintimer_idx = wintimer_len-1;
		int offset_y = wintimer_lut[wintimer_idx];

		// draw score, except winner's flickers ---------@/
		int winner = work->ball.owner;
		int shld_draw = (work->time_frameAbsolute%2);
		if(winner == 0) {
			pgame_fontScore_draw(textbufP2,8,43);
			if(shld_draw) pgame_fontScore_draw(textbufP1,8,80+offset_y);
		} else {
			if(shld_draw) pgame_fontScore_draw(textbufP2,8,43+offset_y);
			pgame_fontScore_draw(textbufP1,8,80);
		}
	}

	if(work->stateID != GAMESTATE_WIN) {
		pgame_fontScore_draw(textbufP2,8,43);
		pgame_fontScore_draw(textbufP1,8,80);
	}

	// draw timer ---------------------------------------@/
	int ms = ((float)(work->time_frame) / 60) * 100;
	sprintf(textbufRoundtime,"%02d:%02d:%02d",
		work->time_mins,
		work->time_seconds,
		ms
	);
	pgame_fontPuchi_draw(textbufRoundtime,16,224);

	peach_drawstate_pop();

	// draw flash over screen, if won -------------------@/
	if(work->stateID == GAMESTATE_WIN) {
		peach_drawstate_push();
		peach_drawstate_blendSet(PEACH_DRAWBLEND_ADD);
		auto dt = (float)(work->flash_timer) / (float)(WIN_FLASHDURATION);
		if(dt > 1) dt = 1;
		int alpha = (int)(WIN_FLASHAMOUNT * (1.0f - dt));
		peach_draw_rect(peach_colorGetRGB(alpha,alpha,alpha),
			0,0,
			SCREEN_X,SCREEN_Y
		);
		peach_drawstate_pop();
	}
}

void pgame_fontPuchi_draw(const char* text, int x, int y) {
	auto work = pgame_workGet();
	auto graph = work->gr_fontPuchi;
	static const int chr_size = 8;

	int dx = x;
	for(int i=0; text[i]; i++) {
		const int chr = text[i];
		static const int FNT_START = 0x20;
		static const int FNT_END = 0x5F;

		// puchi font only stores characters 0x20-0x5F
		if(chr >= FNT_START && chr <= FNT_END) {
			int idx = chr - FNT_START;
			int src_x = (idx & 15) * chr_size;
			int src_y = (idx / 16) * chr_size;
			peach_graph_drawPart(graph,
				src_x,src_y,
				chr_size,chr_size,
				dx,y,
				PEACH_DRAWFLIP_NONE
			);
		}
		if(chr == '\n') {
			dx = x;
			y += chr_size;
		} else {
			dx += chr_size;
		}
	}
}
void pgame_fontScore_draw(const char* text, int x, int y) {
	auto work = pgame_workGet();
	auto graph = work->gr_fontScore;
	static const int chr_size = 24;

	int dx = x;
	for(int i=0; text[i]; i++) {
		const int chr = text[i];
		if(chr >= '0' && chr <= '9') {
			int idx = chr - '0';
			int src_x = idx * chr_size;
			peach_graph_drawPart(graph,
				src_x,0,
				chr_size,chr_size,
				dx,y,
				PEACH_DRAWFLIP_NONE
			);
		}
		if(chr == '\n') {
			dx = x;
			y += chr_size;
		} else {
			dx += chr_size;
		}
	}
}

